''
'' timerisr.bas -- shows how to write an interrupt service
''                 routine (for Programmable Interval Timer (PIT))
''                 entirely in QB
''

defint a-z
'$include: 'c:\cmp\qb45\include\qb.bi'
'$include: '..\bi\boostqb.bi'

'' PIT ports and constants
const PITMODE  = &h43
const PITCNT0  = &h40
const PITFREQ& = &h1234DD               '' dividend

'' Programmable Interrupt Controller (PIC) ports and constants
const PICOCW  = &h20
const PICREQ  = &h20
const PICMASK = &h21
const PICEOI  = &h20

const BIOSDIV& = &h10000                '' 18.2 ticks p/ second

declare sub TmrInit    ()
declare sub TmrDone    ()
declare sub TmrDefRate (rate as integer)
declare sub reprogPIT  (div as long)

dim shared tmrFlag as integer
dim shared divisor as long              '' PIT divisor
dim shared biosCnt as long              '' to mantance BIOS by 18.2 ticks */
dim shared old08 as long                '' preview ISR's address
dim shared myCounter as long            '' counter updated by ISR

'':::
  myCounter = 0                         '' clear

  TmrDefRate 200                        '' 200 ticks p/ second

  TmrInit                               '' reprogram PIT, set new ISR

  print time$                           '' print start time

  do
     pokei2 0, &hB800, (7*256) + num    '' show char on top-left of screen
     num = (num + 1) and 255
  loop while (myCounter < 200 * 3)      '' wait 3 seconds

  print time$                           '' print finish time

  TmrDone                               '' restore PIT and old ISR
  
'':::
sub TmrInit static
  dim ptr as long
  dim regs as RegTypeX               

  if (tmrFlag = 0) then                 '' not installed?
     tmrFlag = -1

     biosCnt = 0                        '' clear BIOS counter

     '' check new divisor
     if ((divisor > BIOSDIV) or (divisor = 0)) then divisor = BIOSDIV

     '' get timerHandler label address
     labelptr ptr                       '' ptr= address of timerHandler
     gosub timerHandler                 '' this line will be skipped
     '' \---> never use GOTO with VBDOS or the next lines will be skipped
                                         
     disableintr                        '' CLI

     '' save current int vector
     regs.ax = &h3508                   '' service 35h (get interrupt vector)
     interruptx &h21, regs, regs
     old08 = makelong(regs.es, regs.bx)

     '' set new int vector
     breaklong ptr, regs.ds, regs.dx    '' ds:dx -> timerHandler
     regs.ax = &h2508                   '' service 25h (set interrupt vector)
     interruptx &h21, regs, regs

     '' allow INT0 to ocurr (just to be sure ;])
     out PICMASK, inp(PICMASK) and &hFE

     '' reprogram PIT with new divisor
     reprogPIT divisor

     enableintr                         '' STI
  end if

  exit sub

'':::
timerHandler:
  pushall                               '' save all registers
  setdgroup                             '' ds= DGROUP
      
  biosCnt = biosCnt - divisor
  if (biosCnt < 0) then                 '' need invoke old ISR?
     biosCnt = biosCnt + BIOSDIV        '' set BIOS counter
      
     pushflags                          '' invoke the preview ISR
     call procedure(old08)              '' /
   
  else
     out PICOCW, PICEOI                 '' send non-specific EOI to PIC 
  end if

  myCounter = myCounter + 1             '' update myCounter

  popall                                '' restore all registers
  intrreturn                            '' return from interrupt
end sub

'':::
sub TmrDone static
  dim regs as RegTypeX

  if (tmrFlag) then                     '' installed?
     tmrFlag = 0

     disableintr                        '' CLI

     '' restore PIT divisor (as defined by BIOS)
     divisor = BIOSDIV                  '' 18.2
     reprogPIT divisor

     '' restore old int vector
     breaklong old08, regs.ds, regs.dx  '' ds:dx -> old08
     regs.ax = &h2508                   '' service 25h (set interrupt vector)
     interruptx &h21, regs, regs

     enableintr                         '' STI
  end if
end sub

'':::
sub TmrDefRate (rate as integer) static
  if (rate > 18) then
     divisor = (PITFREQ \ rate)
  else
     divisor = BIOSDIV                  '' 18.2
  end if

  if (tmrFlag) then reprogPIT(divisor)
end sub

'':::
sub reprogPIT (div as long) static
  '' send command to PIT: counter 0, binary,
  '' rate generation (mode 2), LSB + MSB
  out PITMODE, &h34

  '' set divisor
  out PITCNT0, div and &h000000FF&
  out PITCNT0, LSHR(8, div)             '' \ 256
end sub
