'ͻ
' A library to play MIDI files, based upon QMIDI 4.1 by Jesse Dorland.     
' Requires the SBMIDI driver to be installed. Programmed by IceWolf.       
'ͼ

DEFINT A-Z
' $DYNAMIC

' Declare the SUB's and FUNCTION's.
DECLARE SUB DetectSettings ()
DECLARE SUB DetectDrivers ()
DECLARE SUB InterruptX (IntNum AS INTEGER, InRegs AS ANY, OutRegs AS ANY)
DECLARE SUB LoopMIDI ()
DECLARE SUB LoadMIDI (Filename$)
DECLARE SUB PauseMIDI ()
DECLARE SUB PlayMIDI ()
DECLARE SUB ResumeMIDI ()
DECLARE SUB ReverseStereo (Switch%)
DECLARE SUB SetVolume (Which%, vol%, Mode%)
DECLARE SUB StopMIDI ()
DECLARE SUB UnloadMIDI ()
DECLARE FUNCTION CurrentMIDI$ (Switch%)
DECLARE FUNCTION GetVolume% (Which%, Mode%)
DECLARE FUNCTION MIDIError$ ()
DECLARE FUNCTION MIDIErrorOccured% ()
DECLARE FUNCTION MIDITime! ()
DECLARE FUNCTION MusicDone ()
DECLARE FUNCTION SoundCard$ ()
DECLARE FUNCTION SoundCardType% ()
DECLARE FUNCTION Version$ ()

' Define the CPU registers type, used for interrupts.
TYPE RegTypeX: AX AS INTEGER: BX AS INTEGER: CX AS INTEGER: DX AS INTEGER: BP AS INTEGER: SI AS INTEGER: DI AS INTEGER: FLAGS AS INTEGER: DS AS INTEGER: ES AS INTEGER: END TYPE

' Set the Boolean variables for logic testing.
CONST TRUE = -1, FALSE = 0

' Define the variables to be available in all SUB's and FUNCTION's.
COMMON SHARED SBMIDI, SBSIM, Segment&, BasePort, IRQ, DMA, CardType
COMMON SHARED MIDILoaded, MIDIPlaying, LastError, RevStereo, CurMIDI$
COMMON SHARED TimeMIDI AS SINGLE, PauseTime AS SINGLE, DetectSettingsCalled

' Initialise a couple of variables.
TimeMIDI = -1
CardType = -1

REM $STATIC
FUNCTION CurrentMIDI$ (Switch)
'----------------------------------------------------------------------------
' Returns the name of the MIDI currently loaded. If no MIDI is loaded, it'll
' return a null string and MIDI error 7. If switch is 0, the full filename is
' returned, otherwise the filename is returned in DOS 8:3 format.
'----------------------------------------------------------------------------

    IF MIDILoaded = FALSE THEN LastError = 4: EXIT FUNCTION
   
    SELECT CASE Switch
        CASE 0: CurrentMIDI$ = CurMIDI$
        CASE ELSE
            FOR i = LEN(CurMIDI$) TO 1 STEP -1: IF MID$(CurMIDI$, i, 1) = "\" OR MID$(CurMIDI$, i, 1) = ":" THEN EXIT FOR
            NEXT: i = i + 1: CurrentMIDI$ = MID$(CurMIDI$, i)
    END SELECT

    LastError = 0

END FUNCTION

SUB DetectDrivers STATIC
'----------------------------------------------------------------------------
' Detects whether the SBMIDI and SBSIM drivers are present, and locates their
' IRQ numbers.
'----------------------------------------------------------------------------

    DIM MIDIRegs AS RegTypeX
   
    SBMIDI = 0: SBSIM = 0

    FOR i = &H80 TO &H8F ' Only &H80 to &H8F are searched as it's unlikely
                         ' the driver will be outside this range.
       
        ' Get the Segment and Offset of the Interrupt code
        MIDIRegs.AX = i + 13568: InterruptX &H21, MIDIRegs, MIDIRegs: Segment& = MIDIRegs.ES: Offset& = MIDIRegs.BX
        IF Segment& = 0 THEN GOTO Skip

        ' Check for SBMIDI
        IF SBMIDI = 0 THEN DEF SEG = Segment& - 17: Temp$ = "": FOR j = 1 TO 6: Temp$ = Temp$ + CHR$(PEEK(271 + j)): NEXT: IF Temp$ = "SBMIDI" THEN SBMIDI = i

        ' Check for SBSIM
        IF SBSIM = 0 AND Segment& <> 0 THEN DEF SEG = Segment& - 1: Temp$ = "": FOR j = 1 TO 5: Temp$ = Temp$ + CHR$(PEEK(274 + j)): NEXT: IF Temp$ = "SBSIM" THEN SBSIM = i

Skip:
    NEXT i

    IF SBMIDI = 0 THEN LastError = 1 ELSE LastError = 0
   
END SUB

SUB DetectSettings STATIC
'----------------------------------------------------------------------------
' Checks the BLASTER environment variable to check whether a sound card
' exists, and its settings.
'----------------------------------------------------------------------------
   
    DetectSettingsCalled = TRUE: BasePort = 0: IRQ = 0: DMA = 0: CardType = 0: Settings$ = ENVIRON$("BLASTER")

    FOR i = 1 TO LEN(Settings$) - 1
        SELECT CASE UCASE$(MID$(Settings$, i, 1))
            CASE "T": CardType = VAL(MID$(Settings$, i + 1, 1))
            CASE "A": BasePort = VAL("&H" + LTRIM$(STR$(VAL(MID$(Settings$, i + 1, 3)))))
            CASE "I": IRQ = VAL(MID$(Settings$, i + 1, 2))
            CASE "D": DMA = VAL(MID$(Settings$, i + 1, 2))
        END SELECT
    NEXT i

    IF CardType = 0 THEN LastError = 2 ELSE LastError = 0

END SUB

FUNCTION GetVolume (Which, Mode) STATIC
'----------------------------------------------------------------------------
' Returns the current volume of a channel
'----------------------------------------------------------------------------
'
' Which: 0 = Master Volume, Left Channel
'        1 = Master Volume, Right Channel
'        2 = Voice Volume, Left Channel
'        3 = Voice Volume, Right Channel
'        4 = MIDI Volume, Left Channel
'        5 = MIDI Volume, Right Channel
'        6 = CD Volume, Left Channel
'        7 = CD Volume, Right Channel
'        8 = Line Volume, Left Channel
'        9 = Line Volume, Right Channel
'
' Mode: 0 = Return True Values
'       1 = Return Value between 0 and 31
'       2 = Return Value between 0 and 15
'
' NOTES: The Value Returned For Mode 0 Can Vary Depending On The Sound Card.
'        The Values For Modes 1 & 2 Are Card Independent.
'
   
    IF CardType = 0 AND DetectSettingsCalled THEN LastError = 2: EXIT FUNCTION
    IF CardType = 0 AND NOT DetectSettingsCalled THEN LastError = 3: EXIT FUNCTION
    IF Which% < 0 OR Which% > 5 THEN LastError = 14: EXIT FUNCTION
    IF Mode% < 0 OR Mode% > 2 THEN LastError = 14: EXIT FUNCTION

    SELECT CASE CardType
        CASE 1: LastError = 10
        CASE 3: LastError = 11
        CASE 2, 4, 5   ' {SB PRO}
            w = Which XOR RevStereo: PortNum = &H22 + (w \ 2) * 2: OUT BasePort + 4, PortNum: Volume = INP(BasePort + 5): t = w MOD 2
            Volume = Volume AND (&HF + 225 * t): IF t = 1 THEN Volume = Volume \ 16
            SELECT CASE Mode
                CASE 0: GetVolume = Volume
                CASE 1: GetVolume = Volume * 2
                CASE 2: GetVolume = ((Volume + 1) / 16 * 16) - 1
            END SELECT
        CASE IS >= 6   ' {SB16/AWE32 etc}
            PortNum = &H30 + Which: IF RevStereo THEN PortNum = PortNum XOR 1
            OUT BasePort + 4, PortNum: Volume = INP(BasePort + 5): Volume = Volume \ 8
            SELECT CASE Mode
                CASE 0: GetVolume = Volume
                CASE 1: GetVolume = Volume
                CASE 2: GetVolume = ((Volume + 1) / 32 * 16) - 1
            END SELECT
    END SELECT

    LastError = 0

END FUNCTION

SUB LoadMIDI (Filename$) STATIC
'----------------------------------------------------------------------------
' Loads a MIDI file into base memory.
'----------------------------------------------------------------------------
  
    IF CardType = 0 AND DetectSettingsCalled THEN LastError = 2: EXIT SUB
    IF CardType = 0 AND NOT DetectSettingsCalled THEN LastError = 3: EXIT SUB
    IF SBMIDI = 0 THEN LastError = 1: EXIT SUB
    IF MIDIPlaying = TRUE THEN StopMIDI
    IF MIDILoaded = TRUE THEN UnloadMIDI

    DIM MIDIRegs AS RegTypeX
    DTA$ = SPACE$(44)
   
    ' Check a filename has actually been supplied.
    IF Filename$ = "" THEN LastError = 14: EXIT SUB
   
    ' If filename doesn't have an extension, add one
    IF INSTR(Filename$, ".") = 0 THEN Filename$ = Filename$ + ".MID"

    ' Check the file exists
    ' Set Disk Transfer Address
    MIDIRegs.AX = &H1A00: MIDIRegs.DS = VARSEG(DTA$): MIDIRegs.DX = SADD(DTA$): InterruptX &H21, MIDIRegs, MIDIRegs
    ' See if the file can be found
    tempname$ = Filename$ + CHR$(0): MIDIRegs.AX = &H4E00: MIDIRegs.CX = &H0: MIDIRegs.DS = VARSEG(tempname$): MIDIRegs.DX = SADD(tempname$): InterruptX &H21, MIDIRegs, MIDIRegs
    IF MIDIRegs.FLAGS AND 1 THEN    ' If CF is set, there was an error
        IF MIDIRegs.AX = &H2 OR MIDIRegs.AX = &H12 THEN LastError = 7: EXIT SUB
        IF MIDIRegs.AX = &H3 THEN LastError = 12: EXIT SUB
        LastError = 15: EXIT SUB
    END IF
         
    FF = FREEFILE: OPEN Filename$ FOR BINARY AS #FF: FileLen& = LOF(FF): CLOSE #FF

    MIDIRegs.AX = &H4800: MIDIRegs.BX = (FileLen& \ 16) + 1: InterruptX &H21, MIDIRegs, MIDIRegs: tempname$ = Filename$ + CHR$(0)

    IF MIDIRegs.AX = 7 OR MIDIRegs.AX = 8 THEN
        LargestBlock& = MIDIRegs.BX: LargestBlock& = LargestBlock& * 16: MemUsed& = (FileLen& + 2048) - LargestBlock&: a& = SETMEM(-MemUsed&): MIDIRegs.AX = &H4800: MIDIRegs.BX = (FileLen& \ 16) + 1: InterruptX &H21, MIDIRegs, MIDIRegs
        IF MIDIRegs.AX = 7 OR MIDIRegs.AX = 8 THEN a& = SETMEM(650000): LastError = 8: EXIT SUB
    END IF

    Segment& = MIDIRegs.AX: MIDISegment& = MIDIRegs.AX: MIDIRegs.AX = &H3D00: MIDIRegs.DX = SADD(tempname$): MIDIRegs.DS = VARSEG(tempname$): InterruptX &H21, MIDIRegs, MIDIRegs: Handle = MIDIRegs.AX

    FOR i& = 1 TO FileLen& STEP 16384: MIDIRegs.AX = &H3F00: MIDIRegs.CX = 16384: MIDIRegs.DX = 0: MIDIRegs.DS = VAL("&H" + HEX$(MIDISegment&)): MIDIRegs.BX = Handle: InterruptX &H21, MIDIRegs, MIDIRegs: MIDISegment& = MIDISegment& + 1024: NEXT i&
 
    MIDIRegs.AX = &H3E00: MIDIRegs.BX = Handle: InterruptX &H21, MIDIRegs, MIDIRegs
   
    MIDILoaded = TRUE
    CurMIDI$ = Filename$
    LastError = 0

END SUB

SUB LoopMIDI STATIC
'----------------------------------------------------------------------------
' Determines whether the MIDI has finished, and if so plays it again.
'----------------------------------------------------------------------------
   
    IF CardType = 0 AND DetectSettingsCalled THEN LastError = 2: EXIT SUB
    IF CardType = 0 AND NOT DetectSettingsCalled THEN LastError = 3: EXIT SUB
    IF SBMIDI = 0 THEN LastError = 1: EXIT SUB
    IF MIDILoaded = FALSE THEN LastError = 4: EXIT SUB
    IF MIDIPlaying = FALSE THEN LastError = 6: EXIT SUB

    DIM MIDIRegs AS RegTypeX
  
    MIDIRegs.BX = 11: InterruptX SBMIDI, MIDIRegs, MIDIRegs: IF MIDIRegs.AX = 0 THEN PlayMIDI

    LastError = 0

END SUB

FUNCTION MIDIError$
'----------------------------------------------------------------------------
' Returns an error message generated by the MIDI functions.
'----------------------------------------------------------------------------

    SELECT CASE LastError
        CASE &H0: MIDIError$ = "Operation Successful."
        CASE &H1: MIDIError$ = "The SBMIDI Driver has not be detected."
        CASE &H2: MIDIError$ = "The BLASTER environment variable is not set."
        CASE &H3: MIDIError$ = "DetectSettings has not been called."
        CASE &H4: MIDIError$ = "A MIDI file has not been loaded."
        CASE &H5: MIDIError$ = "A MIDI file is already playing."
        CASE &H6: MIDIError$ = "No MIDI file is playing."
        CASE &H7: MIDIError$ = "The file could not be found."
        CASE &H8: MIDIError$ = "There was insufficient memory to complete the operation."
        CASE &H9: MIDIError$ = "The MIDI is not paused."
        CASE &HA: MIDIError$ = "Your Sound card does not support volume alteration."
        CASE &HB: MIDIError$ = "Support for older sound cards not yet included."
        CASE &HC: MIDIError$ = "Path not found."
        CASE &HD: MIDIError$ = "No filename supplied."
        CASE &HD: MIDIError$ = "A value is out of the valid range."
        CASE &HF: MIDIError$ = "Unknown error."
    END SELECT

END FUNCTION

FUNCTION MIDIErrorOccured
'----------------------------------------------------------------------------
' Returns error number if an error has occured, otherwise FALSE.
'----------------------------------------------------------------------------

    IF LastError <> 0 THEN MIDIErrorOccured = LastError ELSE MIDIErrorOccured = 0

END FUNCTION

FUNCTION MIDITime!
'----------------------------------------------------------------------------
' Returns the length of time the MIDI has been playing in seconds.
' Returns -1 if no MIDI is playing.
'----------------------------------------------------------------------------
    IF MIDIPlaying = 1 THEN
        MIDITime! = PauseTime
    ELSEIF TimeMIDI >= 0 THEN
        CurrentTime! = TIMER
        IF CurrentTime! - TimeMIDI < 0 THEN CurrentTime! = 86400 + CurrentTime!
        MIDITime! = CurrentTime! - TimeMIDI
    ELSE
        LastError = 6
        MIDITime! = -1
    END IF
END FUNCTION

FUNCTION MusicDone STATIC
'----------------------------------------------------------------------------
' Checks whether the MIDI has finished.
'----------------------------------------------------------------------------
   
    IF CardType = 0 AND DetectSettingsCalled THEN LastError = 2: EXIT FUNCTION
    IF CardType = 0 AND NOT DetectSettingsCalled THEN LastError = 3: EXIT FUNCTION
    IF SBMIDI = 0 THEN LastError = 1: EXIT FUNCTION
    IF MIDILoaded = FALSE THEN LastError = 4: EXIT FUNCTION
    IF MIDIPlaying = FALSE THEN LastError = 6: EXIT FUNCTION

    DIM MIDIRegs AS RegTypeX
   
    MIDIRegs.BX = 11: InterruptX SBMIDI, MIDIRegs, MIDIRegs: IF MIDIRegs.AX = 0 THEN MusicDone = -1 ELSE MusicDone = 0

    LastError = 0

END FUNCTION

SUB PauseMIDI STATIC
'----------------------------------------------------------------------------
' Pauses the currently playing MIDI file.
'----------------------------------------------------------------------------
   
    IF CardType = 0 AND DetectSettingsCalled THEN LastError = 2: EXIT SUB
    IF CardType = 0 AND NOT DetectSettingsCalled THEN LastError = 3: EXIT SUB
    IF SBMIDI = 0 THEN LastError = 1: EXIT SUB
    IF MIDIPlaying = FALSE THEN LastError = 6: EXIT SUB

    DIM MIDIRegs AS RegTypeX
 
    MIDIRegs.BX = 7: InterruptX SBMIDI, MIDIRegs, MIDIRegs

    PauseTime = MIDITime!: TimeMIDI = 0: IF PauseTime = 0 THEN PauseTime = .00001
    MIDIPlaying = 1
    LastError = 0

END SUB

SUB PlayMIDI STATIC
'----------------------------------------------------------------------------
' Starts a MIDI playing.
'----------------------------------------------------------------------------
   
    IF CardType = 0 AND DetectSettingsCalled THEN LastError = 2: EXIT SUB
    IF CardType = 0 AND NOT DetectSettingsCalled THEN LastError = 3: EXIT SUB
    IF SBMIDI = 0 THEN LastError = 1: EXIT SUB
    IF MIDILoaded = FALSE THEN LastError = 4: EXIT SUB

    DIM MIDIRegs AS RegTypeX
   
    MIDIRegs.BX = 4
    MIDIRegs.DX = Segment&
    MIDIRegs.AX = 0
    InterruptX SBMIDI, MIDIRegs, MIDIRegs

    MIDIRegs.BX = 5
    InterruptX SBMIDI, MIDIRegs, MIDIRegs
   
    IF MIDIRegs.AX <> 0 THEN LastError = 4: EXIT SUB

    TimeMIDI = TIMER: PauseTime = 0
    MIDIPlaying = TRUE
    LastError = 0

END SUB

SUB ResumeMIDI STATIC
'----------------------------------------------------------------------------
' Resumes the playing of a paused MIDI.
'----------------------------------------------------------------------------
   
    IF CardType = 0 AND DetectSettingsCalled THEN LastError = 2: EXIT SUB
    IF CardType = 0 AND NOT DetectSettingsCalled THEN LastError = 3: EXIT SUB
    IF SBMIDI = 0 THEN LastError = 1: EXIT SUB
    IF MIDILoaded = FALSE THEN LastError = 4: EXIT SUB
    IF MIDIPlaying <> 1 THEN LastError = 9: EXIT SUB
   
    DIM MIDIRegs AS RegTypeX
  
    MIDIRegs.BX = 8: InterruptX SBMIDI, MIDIRegs, MIDIRegs

    TimeMIDI = TIMER - PauseTime: PauseTime = 0
    MIDIPlaying = TRUE
    LastError = 0

END SUB

SUB ReverseStereo (Switch)
'----------------------------------------------------------------------------
' Reverses the left and right stereo channels.
'----------------------------------------------------------------------------
' For some reason, some sound cards send the left channel data to the right
' speaker and vice versa. This sub counteracts this by reversing the left and
' right channels.
'
' If anyone knows of a way to autodetect whether the sound card reverses the
' left and right channels, then send me an e-mail to IceWolf@SimonsMail.Com.
'

    temp0 = GetVolume(0, 0)
    temp1 = GetVolume(1, 0)
    temp2 = GetVolume(2, 0)
    temp3 = GetVolume(3, 0)
    temp4 = GetVolume(4, 0)
    temp5 = GetVolume(5, 0)
    RevStereo = Switch
    SetVolume 0, temp0, 0
    SetVolume 1, temp1, 0
    SetVolume 2, temp2, 0
    SetVolume 3, temp3, 0
    SetVolume 4, temp4, 0
    SetVolume 5, temp5, 0

END SUB

SUB SetVolume (Which, vol, Mode) STATIC
'----------------------------------------------------------------------------
' Sets the volume of a specific type.
'----------------------------------------------------------------------------
'
' Which: 0 = Master Volume, Left Channel
'        1 = Master Volume, Right Channel
'        2 = Voice Volume, Left Channel
'        3 = Voice Volume, Right Channel
'        4 = MIDI Volume, Left Channel
'        5 = MIDI Volume, Right Channel
'
' Mode: 0 = Vol Is A True Value
'       1 = Vol Is A Percentage
'       2 = Vol Is A Value between 0 and 15
'
' NOTES: The Value Used For Mode 0 Can Vary Depending On The Mixer Chip In
'        The Sound Card. The Values For Modes 1 & 2 Are Card Independent. A
'        Value Of -1 Is Returned If Either Parameter Is Out Of Range, Or No
'        Sound Card Is Present.
'

    IF CardType = 0 AND DetectSettingsCalled THEN LastError = 2: EXIT SUB
    IF CardType = 0 AND NOT DetectSettingsCalled THEN LastError = 3: EXIT SUB
    IF Which% < 0 OR Which% > 5 THEN LastError = 14: EXIT SUB
    IF Mode% < 0 OR Mode% > 2 THEN LastError = 14: EXIT SUB

    SELECT CASE CardType
        CASE 1: LastError = 10
        CASE 3: LastError = 11
        CASE 2, 4, 5   ' {SB PRO}
            IF vol < 0 THEN vol = 0
            IF vol > 15 THEN vol = 15
            w = Which XOR RevStereo: PortNum = &H22 + (w \ 2) * 2
            SELECT CASE Mode
                CASE 0: Volume = vol
                CASE 1: Volume = vol \ 2
                CASE 2: Volume = vol
            END SELECT
            IF (w MOD 2) = 0 THEN t = GetVolume((w + 1) XOR RevStereo, 0) ELSE t = GetVolume((w - 1) XOR RevStereo, 0)
            OUT BasePort + 4, PortNum: IF (w MOD 2) = 0 THEN OUT BasePort + 5, t * 16 + Volume ELSE OUT BasePort + 5, Volume * 16 + t
        CASE IS >= 6   ' {SB16/AWE32 etc}
            IF vol < 0 THEN vol = 0
            IF vol > 31 THEN vol = 31
            PortNum = &H30 + Which: IF RevStereo THEN PortNum = PortNum XOR 1
            SELECT CASE Mode
                CASE 0: Volume = vol
                CASE 1: Volume = vol
                CASE 2: Volume = vol * 2
            END SELECT
            OUT BasePort + 4, PortNum: OUT BasePort + 5, Volume * 8
    END SELECT
   
    LastError = 0

END SUB

FUNCTION SoundCard$
'----------------------------------------------------------------------------
' Returns the name of the sound card, eg. Sound Blaster 16/Sound Blaster Pro.
'----------------------------------------------------------------------------

    SELECT CASE CardType
        CASE -1: SoundCard$ = "DetectSettings has not been called."
        CASE 0: SoundCard$ = "No sound card detected"
        CASE 1: SoundCard$ = "Sound Blaster 1.0/1.5"
        CASE 2: SoundCard$ = "Sound Blaster Pro"
        CASE 3: SoundCard$ = "Sound Blaster 2.0"
        CASE 4, 5: SoundCard$ = "Sound Blaster Pro 2"
        CASE 6: SoundCard$ = "Sound Blaster 16/32/AWE32/AWE64"
        CASE ELSE: SoundCard$ = "Sound Card Unknown."
    END SELECT

END FUNCTION

FUNCTION SoundCardType
'----------------------------------------------------------------------------
' Returns the type number of the sound card (note: this is the same as the
' number after the 'T' in the BLASTER environment variable).
'----------------------------------------------------------------------------

    SoundCardType = CardType

END FUNCTION

SUB StopMIDI STATIC
'----------------------------------------------------------------------------
' Stops the MIDI playing.
' exists, and its settings.
'----------------------------------------------------------------------------
   
    IF CardType = 0 AND DetectSettingsCalled THEN LastError = 2: EXIT SUB
    IF CardType = 0 AND NOT DetectSettingsCalled THEN LastError = 3: EXIT SUB
    IF SBMIDI = 0 THEN LastError = 1: EXIT SUB
    IF MIDIPlaying = FALSE THEN LastError = 6: EXIT SUB

    DIM MIDIRegs AS RegTypeX
  
    MIDIRegs.BX = 4: MIDIRegs.DX = Segment&: MIDIRegs.AX = 0: InterruptX SBMIDI, MIDIRegs, MIDIRegs

    TimeMIDI = -1: PauseTime = 0
    MIDIPlaying = FALSE
    LastError = 0

END SUB

SUB UnloadMIDI STATIC
'----------------------------------------------------------------------------
' Frees the memory used by the MIDI and gives it back to QB.
'----------------------------------------------------------------------------
   
    IF CardType = 0 AND DetectSettingsCalled THEN LastError = 2: EXIT SUB
    IF CardType = 0 AND NOT DetectSettingsCalled THEN LastError = 3: EXIT SUB
    IF SBMIDI = 0 THEN LastError = 1: EXIT SUB
    IF MIDILoaded = FALSE THEN LastError = 4: EXIT SUB
    IF MIDIPlaying = TRUE THEN StopMIDI

    DIM MIDIRegs AS RegTypeX
   
    MIDIRegs.ES = Segment&: MIDIRegs.AX = &H4900: InterruptX &H21, MIDIRegs, MIDIRegs: a& = SETMEM(650000)

    MIDILoaded = FALSE
    CurMIDI$ = ""
    LastError = 0

END SUB

FUNCTION Version$
'----------------------------------------------------------------------------
' Returns a string containing the version number of the library.
'----------------------------------------------------------------------------
   
    Version$ = "1.1"

END FUNCTION

